<?php

declare(strict_types=1);

namespace Models;

class WorkoutSession extends Model
{
    protected string $table = 'workout_sessions';

    public function getForUserAndPlan(int $userId, int $planId): array
    {
        $stmt = $this->db->prepare("
            SELECT * FROM {$this->table}
            WHERE user_id = ? AND plan_id = ?
            ORDER BY session_date DESC, day_number ASC
        ");
        $stmt->execute([$userId, $planId]);
        return $stmt->fetchAll();
    }

    public function getByUserAndDate(int $userId, string $date): array
    {
        $stmt = $this->db->prepare("SELECT * FROM {$this->table} WHERE user_id = ? AND session_date = ? ORDER BY day_number");
        $stmt->execute([$userId, $date]);
        return $stmt->fetchAll();
    }

    public function findForUserPlanDayDate(int $userId, int $planId, int $dayNumber, string $sessionDate): ?array
    {
        $stmt = $this->db->prepare("
            SELECT * FROM {$this->table}
            WHERE user_id = ? AND plan_id = ? AND day_number = ? AND session_date = ?
        ");
        $stmt->execute([$userId, $planId, $dayNumber, $sessionDate]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function start(int $userId, int $planId, int $dayNumber, string $sessionDate): int
    {
        $existing = $this->findForUserPlanDayDate($userId, $planId, $dayNumber, $sessionDate);
        if ($existing) {
            return (int) $existing['id'];
        }
        return $this->insert([
            'user_id' => $userId,
            'plan_id' => $planId,
            'day_number' => $dayNumber,
            'session_date' => $sessionDate,
        ]);
    }

    public function complete(int $id, int $userId, ?int $difficultyRating = null, ?string $notes = null): bool
    {
        $session = $this->find($id);
        if (!$session || (int) $session['user_id'] !== $userId) {
            return false;
        }
        $data = ['completed_at' => date('Y-m-d H:i:s')];
        if ($difficultyRating !== null) $data['difficulty_rating'] = $difficultyRating;
        if ($notes !== null) $data['notes'] = $notes;
        return $this->update($id, $data);
    }

    public function getWithSetLogs(int $sessionId): ?array
    {
        $session = $this->find($sessionId);
        if (!$session) return null;
        $setLogModel = new WorkoutSetLog();
        $session['set_logs'] = $setLogModel->getBySessionId($sessionId);
        return $session;
    }
}
