<?php

declare(strict_types=1);

namespace Models;

class ProgressLog extends Model
{
    protected string $table = 'progress_logs';

    public function getByUserId(int $userId, int $limit = 50): array
    {
        $stmt = $this->db->prepare("
            SELECT pl.*, wp.start_date, wp.duration_weeks
            FROM {$this->table} pl
            LEFT JOIN workout_plans wp ON wp.id = pl.plan_id
            WHERE pl.user_id = ?
            ORDER BY pl.workout_date DESC, pl.logged_at DESC
            LIMIT ?
        ");
        $stmt->execute([$userId, $limit]);
        return $stmt->fetchAll();
    }

    public function getByUserAndDate(int $userId, string $date): ?array
    {
        $stmt = $this->db->prepare("SELECT * FROM {$this->table} WHERE user_id = ? AND workout_date = ?");
        $stmt->execute([$userId, $date]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function create(int $userId, string $workoutDate, bool $completed, ?int $planId = null, ?int $difficultyRating = null, ?string $notes = null): int
    {
        return $this->insert([
            'user_id' => $userId,
            'plan_id' => $planId,
            'workout_date' => $workoutDate,
            'completed' => $completed ? 1 : 0,
            'difficulty_rating' => $difficultyRating,
            'notes' => $notes,
        ]);
    }

    public function getStatsForUser(int $userId): array
    {
        $stmt = $this->db->prepare("
            SELECT 
                COUNT(*) AS total_workouts,
                SUM(completed) AS completed_workouts,
                AVG(difficulty_rating) AS avg_difficulty
            FROM {$this->table}
            WHERE user_id = ?
        ");
        $stmt->execute([$userId]);
        return $stmt->fetch();
    }

    /** Consecutive days with completed workouts (from progress_logs + workout_sessions). */
    public function getStreak(int $userId): int
    {
        $stmt = $this->db->prepare("
            (SELECT DISTINCT workout_date AS d FROM {$this->table} WHERE user_id = ? AND completed = 1)
            UNION
            (SELECT DISTINCT session_date AS d FROM workout_sessions WHERE user_id = ? AND completed_at IS NOT NULL)
        ");
        $stmt->execute([$userId, $userId]);
        $dates = array_unique($stmt->fetchAll(\PDO::FETCH_COLUMN));
        rsort($dates);
        $today = date('Y-m-d');
        $streak = 0;
        $expected = $today;
        foreach ($dates as $d) {
            if ($d === $expected) {
                $streak++;
                $expected = date('Y-m-d', strtotime($expected . ' -1 day'));
            } else {
                break;
            }
        }
        return $streak;
    }
}
