<?php

declare(strict_types=1);

namespace Controllers;

use Models\User;
use Models\Profile;

class AuthController extends BaseController
{
    private ?User $userModel = null;
    private ?Profile $profileModel = null;

    private function getUserModel(): User
    {
        if ($this->userModel === null) {
            $this->userModel = new User();
        }
        return $this->userModel;
    }

    private function getProfileModel(): Profile
    {
        if ($this->profileModel === null) {
            $this->profileModel = new Profile();
        }
        return $this->profileModel;
    }

    public function showLogin(): void
    {
        if ($this->isLoggedIn()) {
            $this->redirect('/');
            return;
        }
        $this->view('auth/login', ['title' => 'Login']);
    }

    public function login(): void
    {
        $email = trim($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        if ($email === '' || $password === '') {
            $this->redirect('/login?error=missing');
            return;
        }
        $user = $this->getUserModel()->verifyPassword($email, $password);
        if (!$user) {
            $this->redirect('/login?error=invalid');
            return;
        }
        $_SESSION['user_id'] = (int) $user['id'];
        $_SESSION['user_email'] = $user['email'];
        $_SESSION['role'] = $user['role'] ?? 'user';
        if (($_SESSION['role'] ?? '') === 'admin') {
            $this->redirect('/admin');
            return;
        }
        $this->redirect('/');
    }

    public function showRegister(): void
    {
        if ($this->isLoggedIn()) {
            $this->redirect('/');
            return;
        }
        $this->view('auth/register', ['title' => 'Register']);
    }

    public function register(): void
    {
        $email = trim($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $confirm = $_POST['confirm_password'] ?? '';
        if ($email === '' || $password === '' || $confirm === '') {
            $this->redirect('/register?error=missing');
            return;
        }
        if (strlen($password) < 8) {
            $this->redirect('/register?error=short');
            return;
        }
        if ($password !== $confirm) {
            $this->redirect('/register?error=mismatch');
            return;
        }
        if ($this->getUserModel()->findByEmail($email)) {
            $this->redirect('/register?error=exists');
            return;
        }
        $userId = $this->getUserModel()->create($email, $password);
        $_SESSION['user_id'] = $userId;
        $_SESSION['user_email'] = $email;
        $this->redirect('/profile/edit');
    }

    public function logout(): void
    {
        $_SESSION = [];
        if (ini_get('session.use_cookies')) {
            $params = session_get_cookie_params();
            setcookie(session_name(), '', time() - 42000, $params['path'], $params['domain'], $params['secure'], $params['httponly']);
        }
        session_destroy();
        $this->redirect('/login');
    }

    private function isLoggedIn(): bool
    {
        return !empty($_SESSION['user_id']);
    }
}
